<?php


/**
 * The Elementor functionality of the plugin.
 *
 * @link       https://conicplex.com
 * @since      1.0.0
 *
 * @package    bsccp
 * @subpackage bsccp/admin
 */

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and methods for registering and rendering
 * the custom Elementor widget.
 *
 * @package    bsccp
 * @subpackage bsccp/admin
 * @author     Sajil Memon <hello@conicplex.com>
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class BDPCP_Product_Widget extends Widget_Base
{
    /**
     * The name of the plugin.
     *
     * @since 1.0.0
     * @var string
     */
    private $plugin_name;

    /**
     * The version of the plugin.
     *
     * @since 1.0.0
     * @var string
     */
    private $version;

    /**
     * Constructor for the class.
     *
     * @since 1.0.0
     * @param array $data Data for the widget.
     * @param array|null $args Arguments for the widget, including 'plugin_name' and 'version'.
     */
    public function __construct($data = [], $args = null)
    {
        parent::__construct($data, $args);

        $this->plugin_name = $args['plugin_name'];
        $this->version = $args['version'];
    }

    /**
     * Get the name of the widget.
     *
     * @since 1.0.0
     * @return string The name of the widget.
     */
    public function get_name()
    {
        return 'bdpcp_bookingdaddy_widget';
    }

    /**
     * Get the title of the widget.
     *
     * @since 1.0.0
     * @return string The title of the widget.
     */
    public function get_title()
    {
        return esc_html__('BookingDaddy Pro', 'bdpcp');
    }

    /**
     * Get the icon of the widget.
     *
     * @since 1.0.0
     * @return string The icon of the widget.
     */
    public function get_icon()
    {
        return 'bdpcp_elementor_icon';
    }

    /**
     * Get the category of the widget.
     *
     * @since 1.0.0
     * @return array
     */
    public function get_categories()
    {
        return ['general'];
    }

    function get_bookable_products()
    {
        $products = wc_get_products([
            'limit' => -1,
            'type'  => 'bdpcp_bookable_product',
        ]);

        $options = ['' => 'Select Product'];
        foreach ($products as $product) {
            $options[$product->get_id()] = $product->get_name();
        }

        return $options;
    }

    /**
     * Register the controls for the widget.
     *
     * @since 1.0.0
     */
    protected function register_controls()
    {
        /**
         * Style for elementor fields
         * Enqueue Elementor Widget Control Style
         */
        wp_register_style($this->plugin_name . '-elementor-widget-control-pro', BDPCP_ROOT_URL . 'admin/css/bdpcp-elementor-widget-control-admin.css', [], $this->version, 'all');
        wp_enqueue_style($this->plugin_name . '-elementor-widget-control-pro');

        echo '<style>
                i.bdpcp_elementor_icon {
                    display: block !important;
                    width: 26px !important;
                    height: 26px !important;
                    background-image: url("' . BDPCP_ROOT_URL . 'assets/bdpcp-elementor-icon.svg") !important;
                    background-size: contain !important;
                    background-repeat: no-repeat !important;
                    margin: auto !important;
                }
            </style>';

        /* Start Content Section */
        $this->start_controls_section(
            'bdpcp_booking_form_section',
            [
                'label' => esc_html__('Booking Form', 'bdpcp'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bdpcp_product_id',
            [
                'label'   => __('Select Product', 'bdpcp'),
                'type'    => Controls_Manager::SELECT,
                'options' => $this->get_bookable_products(),
                'default' => '',
            ]
        );
        $this->add_control(
            'bdpcp_form_background_color',
            [
                'label' => esc_html__('Background Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_form_border_color',
            [
                'label' => esc_html__('Border Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_form_border_radius',
            [
                'label' => esc_html__('Border Radius:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container' => 'border-radius: {{VALUE}}px',

                ],
                'description' => esc_html__('Enter the border radius in px', 'bdpcp'),

            ]
        );
        $this->end_controls_section();
        /* End pagination style Section */

        /* Start Booking Form Label Section */
        $this->start_controls_section(
            'bdpcp_booking_form_label_section',
            [
                'label' => esc_html__('Form Control', 'bdpcp'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'bdpcp_date_selection_label',
            [
                'label' => esc_html__('Date Selection Label', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->add_control(
            'bdpcp_time_selection_label',
            [
                'label' => esc_html__('Time Selection Label', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->add_control(
            'bdpcp_duration_label',
            [
                'label' => esc_html__('Duration Label', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->add_control(
            'bdpcp_sub_total_label',
            [
                'label' => esc_html__('Sub Total Label', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->add_control(
            'bdpcp_additional_fee_label',
            [
                'label' => esc_html__('Additional Fee Label', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->add_control(
            'bdpcp_total_amount_label',
            [
                'label' => esc_html__('Total Amount Label', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );

        $this->add_control(
            'bdpcp_form_label_color',
            [
                'label' => esc_html__('Label Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-form-label' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-price-item' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_form_label_size',
            [
                'label' => esc_html__('Label Size:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-form-label' => 'font-size: {{VALUE}}px',
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-price-item' => 'font-size: {{VALUE}}px',
                ],
                'description' => esc_html__('Enter the font size in px', 'bdpcp'),
            ]
        );
        $this->add_control(
            'bdpcp_form_control_background_color',
            [
                'label' => esc_html__('Background Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-form-control' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-time-select' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_form_control_text_color',
            [
                'label' => esc_html__('Text Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-form-control' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-time-select' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_form_control_border_radius',
            [
                'label' => esc_html__('Border Radius:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-form-control' => 'border-radius: {{VALUE}}px',
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-time-select' => 'border-radius: {{VALUE}}px',
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-btn-book-now' => 'border-radius: {{VALUE}}px',
                ],
                'description' => esc_html__('Enter the border radius in px', 'bdpcp'),
            ]
        );
        $this->end_controls_section();
        /* Start Booking Form Control Section */

        /* Start Booking Button Section */
        $this->start_controls_section(
            'bdpcp_booking_form_book_button_section',
            [
                'label' => esc_html__('Book Button', 'bdpcp'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bdpcp_book_button_text',
            [
                'label' => esc_html__('Text', 'bdpcp'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->add_control(
            'bdpcp_book_button_background_color',
            [
                'label' => esc_html__('Background Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-btn-book-now' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_book_button_text_color',
            [
                'label' => esc_html__('Text Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-btn-book-now' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_section();
        /* Start Booking Button Section */

        /* Start Booking Unit Type Section */
        $this->start_controls_section(
            'bdpcp_booking_unit_type_section',
            [
                'label' => esc_html__('Unit Type', 'bdpcp'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bdpcp_unit_type_background_color',
            [
                'label' => esc_html__('Background Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-people-type-count-display' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bdpcp_unit_type_text_color',
            [
                'label' => esc_html__('Text Color:', 'bdpcp'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .bdpcp-main-container .bdpcp-people-type-count-display' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_section();
        /* Start Booking Unit Type Section */
    }

    /**
     * Rander the widget output.
     *
     * @since 1.0.0
     * @return array
     */
    protected function render()
    {
        // Include elementor widget template
        $bdpcp_elementor_settings = $this->get_settings_for_display();
        include_once BDPCP_ROOT_PATH . 'public/partials/bdpcp-public-display.php';
    }

    /**
     * Add style dependency for widget template.
     *
     * @since 1.0.0
     * @return array
     */
    public function get_style_depends(): array
    {
        return [$this->plugin_name . '-elementor'];
    }
}
